<?php
class Wpjobboard_Si
{
    /**
     * Wpjobboard_Si Singleton
     * 
     * @var Wpjobboard_Si
     */
    protected static $_instance = null;
    
    /**
     * Path to wpjobboard-am (eg. /public_html/wp-content/plugins/wpjobboard-si)
     *
     * @var string
     */
    protected $_basedir = null;
    
    /**
     * URL to wpjobboard-am (eg. http://example.com/wp-content/plugins/wpjobboard-si)
     *
     * @var string
     */
    protected $_baseurl = null;
    
    /**
     * Admin Object
     *
     * @var Wpjobboard_Si_Admin
     */
    public $admin = null;
    

    /**
     * Object contructor
     * 
     * Protected constructor allows to create only one instance of this class.
     * 
     * @since 1.0
     * @return Wpjobboard_Si
     */
    protected function __construct() {
        
        $this->_basedir = dirname( dirname( __FILE__ ) );
        $this->_baseurl = plugin_dir_url( dirname( __FILE__ ) );
        
        add_action( "init", array( $this, "init" ));
        
        if( is_admin() ) {
            include_once $this->_basedir . "/includes/class-wpjobboard-si-admin.php";
            $this->admin = new Wpjobboard_Si_Admin();
        } 
        
        add_action( "wpjb_payment_created", array( &$this, "wpjb_payment_created"), 10, 3 );
        add_action( "wpjb_payment_complete", array( &$this, "wpjb_payment_complete" ), 10, 2 );
        add_action( "wpjb_tpl_pricing_history_actions", array( &$this, "add_invoice_url" ) );  
        
        
    }
    
    /**
     * Singleton
     * 
     * Creates if NULL and returns Wpjobboard_Am instance
     * 
     * @since 1.0
     * @return Wpjobboard_Si
     */
    public static function get_instance() {
        if( self::$_instance === null ) {
            self::$_instance = new self();
        }
        
        return self::$_instance;
    }
    
    /**
     * Returns basedir path
     * 
     * @since 1.0
     * @return string   Basedir path (eg. /public_html/wp-content/plugins/wpjobboard-si)
     */
    public function get_basedir() {
        return $this->_basedir;
    }
    
    /**
     * Returns plugin base URL
     * 
     * @since 1.0
     * @return string   Basedir path (eg. http://example.com/wp-content/plugins/wpjobboard-si)
     */
    public function get_baseurl() {
        return $this->_baseurl;
    }
    
    /**
     * General Init Function
     * 
     * This function is executed in both frontend and wp-admin
     * 
     * @since 1.0
     * @return void
     */
    public function init() {
        add_filter("wpjb_email_template_customs", array( &$this, "wpjb_email_template_customs" ), 10, 2);
        add_filter("wpjb_email_template_objects", array( &$this, "wpjb_email_template_objects" ), 10, 2);
        
        add_filter( 'sliced_invoice_totals_output', array( &$this, 'sliced_invoice_totals_output' ) );
        
        //add_filter('wpjb_message_load_template', array( &$this, 'wpjb_message_load_template'), 10, 2 );
        
        add_filter('wpjb_form_init_payment_default', array( &$this, 'wpjb_form_init_payment_default') );
        add_filter('wpjb_form_init_payment_stripe', array( &$this, 'wpjb_form_init_payment_default') );
    }
    
    /**
     * Loads textdomain
     */
    public function wpjb_load_textdomain() {
        //include_once $this->_basedir . "/includes/class-wpjobboard-si-form.php";
        load_plugin_textdomain("wpjobboard-si", false, plugin_basename(dirname( __FILE__ )));
    }
    
    /**
    * Adds custom fields to e-mail templates
    * @param array $customs    list of custom variables
    * @param string $name      name of current e-mail template
    * @return array            list of custom variables
    */
    function wpjb_em_email_template_customs($customs, $name) {

        if($name == "notify_employer_new_invoice") {
            $customs = array(
                "invoice_url" => __("URL to Invoice", "wpjobboard-si"),
            );
        }    

        return $customs;
    }

    /**
     * Adds objects to e-mail templates
     * @param array $objects    list of objects
     * @param string $name      name of current e-mail template
     * @return array            list of objects
     */

    function wpjb_em_email_template_objects($objects, $name) {

        if( $name == "notify_employer_new_invoice" ) {
            $objects = array( "payment" );
        }

        return $objects;
    }
    
    
    
    /**
     * Creates new Invoice in Sliced Invoices
     * 
     * @param Wpjb_Model_Payment $payment
     * @param type $form
     * @param type $gateway
     */
    public function wpjb_payment_created( $payment, $form, $gateway ) {
        
        // Get Client
        $user = get_user_by("email", $payment->email);

        if(!$user) {
            $fullname = split(" ", $payment->fullname);

            $userdata = array(
                'user_login' 	=> sanitize_text_field( $payment->email ),
                'user_pass'  	=> sanitize_text_field( wp_generate_password() ),
                'user_email' 	=> sanitize_text_field( $payment->email ),
                'first_name' 	=> sanitize_text_field( $fullname[0] ),
            );

            if(count($fullname) > 1) {
                $userdata['last_name'] = sanitize_text_field( $fullname[1] );
            }

            $user_id = wp_insert_user( $userdata );
        } else {
            $user_id = $user->id;
        }

        // Get Company Info
        /*$company = Wpjb_Model_Company::current();
        $address = $company->locationToString();
        
        $company_name = $payment->fullname;
        if($company != null) {
            $company_name = $company->company_name;
        } 

        // Save user info for invoices
        update_user_meta( $user_id, '_sliced_client_business', sanitize_text_field( $company_name ) );
        update_user_meta( $user_id, '_sliced_client_address', wp_kses_post( $address ) );*/
     
        //$company_name = Daq_Request::getInstance()->post('wpjb_si_client');
        $company_name = $form->getElement('wpjb_si_client')->getValue();
        
        update_user_meta( $user_id, '_sliced_client_business', sanitize_text_field( $company_name ) );
        update_user_meta( $user_id, '_sliced_client_address', wp_kses_post( $form->getElement('wpjb_si_address')->getValue() ) );
        update_user_meta( $user_id, '_sliced_client_extra_info', wp_kses_post( $form->getElement('wpjb_si_additional')->getValue() ) );

        // Generate Invoice 
        $invoiceData = array(
            'post_title'    => wp_strip_all_tags('WPJobBoard Invoice for: ' . $company_name),
            'post_content'  => '',
            'post_status'   => 'publish',
            'post_type'     => 'sliced_invoice',
        );

        $invoice = wp_insert_post($invoiceData);
        $pricing = new Wpjb_Model_Pricing($payment->pricing_id);
        
        $taxer = new Wpjb_Utility_Taxer();
        $taxer->setPrice($pricing->price);
        $taxer->setDiscount($payment->payment_discount);
        
        $item = array(
            array(
                'qty'       => 1,
                'title'     => __( 'WPJobBoard Payment', 'wpjobboard-si' ),
                'amount'    => $taxer->value->subtotal + $taxer->value->discount, //$taxer->getTotal($taxer->getPrice()), //$payment->getPrice() * ( ( 100 - sliced_get_tax_amount() ) / 100 ),
                'taxable'   => 'on',
            ),
        );
        
        $invoice_payments = array(
            array(
                'date' => time(),
                'amount' => $taxer->value->subtotal + $taxer->value->tax,
                'status' => 'success',
                'memo' => sprintf( __( 'Payment Gateway: %s', "wpjobboard-si" ), $payment->engine),
            ),
        );
        
        $final_discount = $taxer->value->discount;
        
        if($taxer->isEnabled() && $payment->payment_discount > 0) {
            if($taxer->getType() == 'gross') {
                $item[0]['amount'] = round($taxer->value->price * 100 / ( 100 + $taxer->value->rate ), 2);
            } else {
                $invoice_payments[0]['amount'] = $taxer->value->subtotal + ( $taxer->value->price * ( $taxer->value->rate / 100 ) );
                //$final_discount = ( $taxer->value->price + ( $taxer->value->price * ( $taxer->value->rate / 100 ) ) * $taxer->value-> );
            }
        }
        
        
        if($payment->object_type == 1) {
            // adding job
            $object = new Wpjb_Model_Job($payment->object_id);
            $item[0]['title'] = sprintf( __( 'WPJobBoard Job: %s', 'wpjobboard-si' ), $object->job_title );
            $invoice_description = sprintf( __("Invoice for adding job on %s job board"), get_bloginfo('name') );
        } elseif($payment->object_type == 2) {
            // resume access
            $object = new Wpjb_Model_Resume($payment->object_id);
            $item[0]['title'] = sprintf( __( 'WPJobBoard Resume Access: %s', 'wpjobboard-si' ), $object->getSearch(true)->fullname );
            $invoice_description = sprintf( __("Invoice for access to resume on %s job board"), get_bloginfo('name') );
        } elseif($payment->object_type == 3) {
            // membership
            $object = new Wpjb_Model_Pricing($payment->pricing_id);
            $item[0]['title'] = sprintf( __( 'WPJobBoard Membership: %s', 'wpjobboard-si' ), $object->title );
            $invoice_description = sprintf( __("Invoice for membership on %s job board"), get_bloginfo('name') );
        }
        
        apply_filters("wpjb_si_invoice_items", $item);

        // Get Sliced Invoices Config
        $si_options = get_option('sliced_invoices');

        // Set params for Invoice
        add_post_meta($invoice, '_sliced_client', $user_id );
        add_post_meta($invoice, '_sliced_invoice_prefix', $si_options['prefix'] );
        add_post_meta($invoice, '_sliced_invoice_number', $si_options['number'] );
        if($taxer->isEnabled()) {
            add_post_meta($invoice, '_sliced_tax', $taxer->value->rate ); //$taxer->getTaxRate() /*sliced_get_tax_amount()*/);
        } else {
            add_post_meta($invoice, '_sliced_tax', 0 );
        } 
        add_post_meta($invoice, 'sliced_invoice_discount', $final_discount );
        add_post_meta($invoice, '_sliced_currency_symbol', sliced_get_currency_symbol() );
        add_post_meta($invoice, '_sliced_items', $item );
        add_post_meta($invoice, '_sliced_payment', $invoice_payments );
        add_post_meta($invoice, '_sliced_description', $invoice_description );
        add_post_meta($invoice, '_sliced_totals_for_ordering', $taxer->value->total );//$taxer->getTotal($taxer->getPrice()) /*sliced_get_currency_symbol() . round($payment->getPrice() + $payment->getPrice() * (sliced_get_tax_amount() / 100), 2)*/ );
        add_post_meta($invoice, '_sliced_invoice_terms', $si_options['terms'] );
        add_post_meta($invoice, '_sliced_invoice_created', time() );
        add_post_meta($invoice, '_edit_last', wp_get_current_user()->ID );
        add_post_meta($invoice, '_edit_lock', time() . ":" . wp_get_current_user()->ID );
        add_post_meta($invoice, '_sliced_wpjb_type', $taxer->isEnabled() . ";" . $taxer->getType() );

        wp_set_object_terms($invoice, 'unpaid', 'invoice_status');
        
        apply_filters("wpjb_si_invoice_params", $invoice);

        // Increase Invoice number for future use
        $this_number = $si_options['number'];
        $length     = strlen( (string)$this_number ); // get the length of the number
        $new_number = (int)$this_number + 1; // increment number
        $number     = zeroise( $new_number, $length ); // return the new number, ensuring correct length (if using leading zeros)
        // set the number in the options as the new, next number and update it.
        $si_options['number'] = (string)$number;
        update_option( 'sliced_invoices', $si_options );
        
        // Send e-mail notification
        $mail = Wpjb_Utility_Message::load("notify_employer_new_invoice");
        $mail->setTo($payment->email);
        $mail->assign("payment", $payment);
        $mail->assign("invoice_url", get_permalink($invoice));
        $mail->send();
        
        // Save Invoice ID for pyament
        Wpjb_Model_MetaValue::import("payment", "invoice_id", $invoice, $payment->id);
    }
    
    /**
     * Generate Invoice when payment is completed
     * 
     * @param type $payment
     * @param type $object
     */
    function wpjb_payment_complete($payment, $object) {

        $invoice = Wpjb_Model_MetaValue::getSingle("payment", "invoice_id", $payment->id, true);
        wp_set_object_terms($invoice, 'paid', 'invoice_status');
        add_post_meta($invoice, '_sliced_invoice_due', time());
    }
    
    /**
     * Adds link to Invoice in pricing history menu
     * 
     * @param int $payment_id
     */
    public function add_invoice_url($payment_id) {
        
        $invoice_id = Wpjb_Model_MetaValue::getSingle('payment', 'invoice_id', $payment_id, true);
        ob_start() ?>
            <?php if(isset($invoice_id)): ?>
            | <a target="_blank" href="<?php echo get_permalink($invoice_id) ?>"><?php _e("Invoice", "wpjobboard-si"); ?></a>
            <?php endif; ?>
        <?php 
        $link = ob_get_clean();
        
        echo $link;
    }

    public function sliced_invoice_totals_output( $output ) {
        
        $translate = get_option( 'sliced_translate' );
        
        ob_start();

        if( function_exists('sliced_woocommerce_get_order_id') ) {
            $order_id = sliced_woocommerce_get_order_id( get_the_ID() );
            if ( $order_id ) {
                $output = ob_get_clean();
                echo $output;
                return;
            }
        }
        $taxer  = get_post_meta( get_the_id(), '_sliced_wpjb_type', true );
        list($taxer_enabled, $taxer_type) = explode(";", $taxer);
        
        $totals = Sliced_Shared::get_totals( get_the_id() );
        $after_discount = $totals['sub_total'] - (float)$totals['discounts'];
        $totals['sub_total_taxable'] = $after_discount;
        $tax_amount  = get_post_meta( get_the_id(), '_sliced_totals_for_ordering', true ) - $after_discount;
        
        // Recalculate sub_total and tax
        if($taxer_enabled && $taxer_type == 'gross') {
            $tax_rate = get_post_meta( get_the_id(), '_sliced_tax', true ); 
            $whole = $after_discount + $tax_amount;
            $after_discount = ($whole * 100) / ( 100 + $tax_rate);
            $tax_amount = $whole - $after_discount;
        }
        ?>
            
        <table class="table table-sm table-bordered">
            <tbody>
                <tr class="row-sub-total">
                    <td class="rate"><?php echo ( isset( $translate['sub_total'] ) ? $translate['sub_total'] : __( 'Sub Total', 'sliced-invoices') ); ?></td>
                    <td class="total"><?php _e( sliced_get_invoice_sub_total() ) ?></td>
                </tr>
                
                <?php if ( $totals['discounts'] ): ?>
                    <?php $discount = Sliced_Shared::get_formatted_currency( $totals['discounts'] ); ?>
                    <tr class="row-discount">
                        <td class="rate"><?php echo ( isset( $translate['discount'] ) ? $translate['discount'] : __( 'Discount', 'sliced-invoices') ); ?></td>
                        <td class="total"><span style="color:red;">-<?php echo esc_html( $discount ) ?></span></td>
                    </tr>
                    
                    <tr class="row-discount">
                        <td class="rate"><?php _e( 'After Discount', 'sliced-invoices' ); ?></td>
                        <td class="total"><span><?php echo Sliced_Shared::get_formatted_currency( $after_discount ); ?></span></td>
                    </tr>
                <?php endif; ?>

                <?php if($tax_amount > 0): ?>  
                <tr class="row-tax">
                    <td class="rate"><?php _e( sliced_get_tax_name() ) ?></td>
                    <td class="total"><?php echo Sliced_Shared::get_formatted_currency($tax_amount) //_e( sliced_get_invoice_tax() ) ?></td>
                </tr>
                <?php endif; ?>

                <?php if ( $totals['payments'] ): ?>
                    <?php //$paid = Sliced_Shared::get_formatted_currency( $totals['payments'] ); ?>
                    <tr class="row-paid">
                        <td class="rate"><?php _e( 'Paid', 'sliced-invoices' ) ?></td>
                        <td class="total"><span style="color:red;">-<?php echo Sliced_Shared::get_formatted_currency(get_post_meta( get_the_id(), '_sliced_totals_for_ordering', true )); //esc_html( $paid ) ?></span></td>
                    </tr>
                <?php endif; ?>

                <?php if ( $totals['payments'] || $totals['discounts'] ): ?>
                    <?php $total_due = Sliced_Shared::get_formatted_currency( $totals['total_due'] ); ?>
                    <tr class="table-active row-total">
                        <td class="rate"><strong><?php echo ( isset( $translate['total_due'] ) ? $translate['total_due'] : __( 'Total Due', 'sliced-invoices') ); ?></strong></td>
                        <td class="total"><strong><?php echo esc_html( $total_due ) ?></strong></td>
                    </tr>
                <?php else: ?>
                    <tr class="table-active row-total">
                        <td class="rate"><strong><?php echo ( isset( $translate['total_due'] ) ? $translate['total_due'] : __( 'Total Due', 'sliced-invoices') ); ?></strong></td>
                        <td class="total"><strong><?php _e( sliced_get_invoice_total() ) ?></strong></td>
                    </tr>
                <?php endif; ?>
            </tbody>

        </table>
        
        <?php
        
        $output = ob_get_clean();
        
        return $output;
    }
    
    public function wpjb_form_init_payment_default($form) {
        
        $client     = "";
        $address    = "";
        $additional = "";
                
        $object = Wpjb_Model_Company::current();
        if( isset( $object ) && !empty( $object ) ) {
            
            $company_form = new Wpjb_Form_Frontend_Company($object->id);
            
            $id = "wpjb_session_".str_replace("-", "_", wpjb_transient_id());
            $transient = wpjb_session()->get($id);
            $job_form = new Wpjb_Form_AddJob($transient['job_id']);
            
            $name_field = wpjb_conf( 'wpjb_si_emp_client_name', null );
            if($name_field == 'full_name') {
                $user = get_userdata($object->user_id);
                $client = $user->first_name . " " . $user->last_name;
            } elseif( strpos($name_field, 'company__') !== false && $company_form->hasElement( str_replace("company__", "", $name_field ) ) ) {
                $client = $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue();
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) ) {
                $client = $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue();
            }
            
            $name_field = wpjb_conf( 'wpjb_si_emp_address' ) ;
            if( strpos($name_field, 'company__') !== false && $company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                //$address .= $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() . PHP_EOL;
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0) {
               // $address .= $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() . PHP_EOL;
            }
            
            $location_02 = array();
            $name_field = wpjb_conf( 'wpjb_si_emp_zip_code' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue();
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue();
            }
            $name_field = wpjb_conf( 'wpjb_si_emp_city' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue();
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue();
            }
            $name_field = wpjb_conf( 'wpjb_si_emp_state' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue();
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $location_02[] = $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue();
            }
            
            if(count($location_02) > 0) {
                $address .= join(', ', $location_02) . PHP_EOL;
            }
            
            $name_field = wpjb_conf( 'wpjb_si_emp_country' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $country = Wpjb_List_Country::getByCode( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() );
                $country = trim($country['name']);
                $address .= $country . PHP_EOL;
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $country = Wpjb_List_Country::getByCode( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() );
                $country = trim($country['name']);
                $address .= $country . PHP_EOL;
            }
            
            $name_field = wpjb_conf( 'wpjb_si_emp_ai_tax_id' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $additional .= __("Tax ID: ", "wpjobboard-si") . $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() . PHP_EOL;
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $additional .= __("Tax ID: ", "wpjobboard-si") . $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() . PHP_EOL;
            }
            
            $name_field = wpjb_conf( 'wpjb_si_emp_ai_phone' ) ;
            if( strpos($name_field, 'company__') !== false &&$company_form->hasElement( str_replace("company__", "", $name_field ) ) && strlen( $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() ) > 0 ) {
                $additional .= __("Company Phone: ", "wpjobboard-si") . $company_form->getElement( str_replace("company__", "", $name_field ) )->getValue() . PHP_EOL;
            } elseif( strpos($name_field, 'job__') !== false && $job_form->hasElement( str_replace("job__", "", $name_field ) ) && strlen( $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() ) > 0 ) {
                $additional .= __("Company Phone: ", "wpjobboard-si") . $job_form->getElement( str_replace("job__", "", $name_field ) )->getValue() . PHP_EOL;
            }
            
        }
        
        $form->addGroup( 'invoice', __( "Billing Details", "wpjobboard-si" ) );
        
        $e = new Daq_Form_Element_Text('wpjb_si_client');
        $e->setLabel( __( "Company Name", "wpjobboard-si" ) );
        $e->setValue($client);
        $e->setAttr("placeholder", __( "Company name visible on the invoice.", "wpjobboard-si" ) );
        $form->addElement($e, 'invoice');
        
        $e = new Daq_Form_Element_Textarea('wpjb_si_address');
        $e->setLabel( __( "Company Address", "wpjobboard-si" ) );
        $e->setValue($address);
        $e->setAttr("placeholder", __( "Company billing address ...", "wpjobboard-si" ) );
        $e->setAttr("style", "height:130px");
        
        $form->addElement($e, 'invoice');
        
        $e = new Daq_Form_Element_Textarea('wpjb_si_additional');
        $e->setLabel( __( "Additional Informations", "wpjobboard-si" ) );
        $e->setValue($additional);
        $e->setAttr("placeholder", __( "Additional company informations like e-mail, phone or Tax ID.", "wpjobboard-si" ) );
        $e->setAttr("style", "height:130px");
        $form->addElement($e, 'invoice');
        
        return $form;
    }
}
