<?php

// Bail if called directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Wpjobboard_Mailchimp
{
    /**
     * Wpjobboard_Mailchimp Singleton
     *
     * @var Wpjobboard_Mailchimp
     */
    protected static $_instance = null;

    /**
     * Path to wpjobboard-two (eg. /public_html/wp-content/plugins/wpjobboard-mailchimp)
     *
     * @var string
     */
    protected $_basedir = null;

    /**
     * URL to wpjobboard-two (eg. http://example.com/wp-content/plugins/wpjobboard-mailchimp)
     *
     * @var string
     */
    protected $_baseurl = null;


    /**
     * Object contructor
     *
     * Protected constructor allows to create only one instance of this class.
     *
     * @since 1.0
     * @return Wpjobboard_Mailchimp
     */
    protected function __construct() {

        $this->_basedir = dirname( dirname( __FILE__ ) );
        $this->_baseurl = plugin_dir_url( dirname( __FILE__ ) );
        add_action( "init", array( $this, "init" ) );
        
        add_action( 'admin_enqueue_scripts',  array( $this, 'load_admin_style' ) );
        
        if(is_admin()) {
            add_action( "plugins_loaded", array( $this, "updates_manager" ) );
        }

    }

    /**
     * Singleton
     *
     * Creates if NULL and returns Wpjobboard_Mailchimp instance
     *
     * @since 1.0
     * @return Wpjobboard_Mailchimp
     */
    public static function get_instance() {
        if( self::$_instance === null ) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

    /**
     * Returns basedir path
     *
     * @since 1.0
     * @return string   Basedir path (eg. /public_html/wp-content/plugins/wpjobboard-mc)
     */
    public function get_basedir() {
        return $this->_basedir;
    }

    /**
     * Returns plugin base URL
     *
     * @since 1.0
     * @return string   Basedir path (eg. http://example.com/wp-content/plugins/wpjobboard-mc)
     */
    public function get_baseurl() {
        return $this->_baseurl;
    }

    /**
     * General Init Function
     *
     * This function is executed in both frontend and wp-admin
     *
     * @since 1.0
     * @return void
     */
    public function init() {
        
        add_filter( "wpjb_config_sections", array( &$this, "wpjb_config_sections" ) );
          
        add_filter( "wpjr_form_init_register", array( &$this, "wpjr_form_init_register" ) );
        add_filter( "wpjb_form_init_company", array( &$this, "wpjr_form_init_company" ) );
        
        add_filter( "wpjr_form_save_register", array( &$this, "wpjr_form_save_register" ) );
        add_filter( "wpjb_form_save_company", array( &$this, "wpjb_form_save_company" ) );
        
        add_action( 'wp_ajax_wpjb_mc_get_ids', array( &$this, 'wpjb_mc_get_ids' ) );
        add_action( 'wp_ajax_wpjb_mc_add_to_list', array( &$this, 'wpjb_mc_add_to_list' ) );
    }
    
    /**
     * Adds config option in Seetings (WPJB)
     * 
     * @param Array $config
     * @return Array
     */
    function wpjb_config_sections($config) {
        
        include_once $this->get_basedir() . "/includes/class-wpjobboard-mc-form.php";
        
        $config[1]['item'][] = array(
            "form"  => "mailchimp",
            "class" => "Wpjobboard_Mailchimp_Config_Form",
            "image" => $this->get_baseurl() ."/assets/images/mailchimp.png",
            "icon"  => "wpjb-icon-puzzle",
            "title" => __("MailChimp", "wpjobboard-mc"),
            "order" => 10,
        );

        return $config;
    }
    
    function wpjr_form_init_register( $form ) {
        
        $listID = wpjb_conf( "mailchimp_candidate_list", null );
        
        if( $listID == null ) {
            return $form;
        }
        
        $form->addGroup( "newsletter", __("Newsletter", "wpjobboard-mc" ) );
        
        $e = new Daq_Form_Element_Checkbox('newsletter_subscribtion');
        $e->setLabel( __( "Newsletter", "wpjobboard-mc" ) );
        $e->addOption( 1, 1, __( "Subscribe for newsletter", "wpjb-mc" ) );
        $e->setBuiltin( false );
        $form->addElement( $e, "newsletter" );
        
        return $form;
    }
    
    function wpjr_form_init_company( $form ) {
        
        $listID = wpjb_conf( "mailchimp_employer_list", null );
        
        if( $listID == null ) {
            return $form;
        }
        
        $form->addGroup( "newsletter", __("Newsletter", "wpjobboard-mc" ) );
        
        $e = new Daq_Form_Element_Checkbox('newsletter_subscribtion');
        $e->setLabel( __( "Newsletter", "wpjobboard-mc" ) );
        $e->addOption( 1, 1, __( "Subscribe for newsletter", "wpjb-mc" ) );
        $e->setBuiltin( false );
        $form->addElement( $e, "newsletter" );
        
        return $form;
    }
    
    function wpjr_form_save_register( $form ) {
        
        $listID = wpjb_conf( "mailchimp_candidate_list", null );
        
        if( $listID == null ) {
            return $form;
        }
        
        $sub = Daq_Request::getInstance()->post("newsletter_subscribtion", null );
        if( $sub[0] != 1 ) {
            return $form;
        }
        
        $email = $form->getObject()->getUser( true )->user_email;
        $fname = $form->getObject()->getUser( true )->first_name;
        $lname = $form->getObject()->getUser( true )->last_name;
        
        $this->add_user_to_list( $listID, $email, $fname, $lname);
        
        return $form;
    }
    
    function wpjb_form_save_company( $form ) {
        
        $listID = wpjb_conf( "mailchimp_employer_list", null );
        
        if( $listID == null ) {
            return $form;
        }
        
        $sub = Daq_Request::getInstance()->post("newsletter_subscribtion", null );
        if( $sub[0] != 1 ) {
            return $form;
        }
        
        $email = $form->getElement("user_email")->getValue();
        $fname = $form->getObject()->company_name;
        $lname = __( "Company", "wpjb-sb" );
        
        $this->add_user_to_list( $listID, $email, $fname, $lname);
        
        return $form;
    }

    private function add_user_to_list( $listID, $email, $fname, $lname ) {
        $apiKey = wpjb_conf( "mailchimp_api_key" );
        
        // MailChimp API URL
        $memberID = md5( strtolower( $email ) );
        $dataCenter = substr($apiKey,strpos($apiKey,'-')+1);
        $url = 'https://' . $dataCenter . '.api.mailchimp.com/3.0/lists/' . $listID . '/members/' . $memberID;
        
        // member information
        $array = array(
            'email_address' => $email,
            'status'        => 'subscribed',
            'merge_fields'  => array( 
                'FNAME'     => (string)$fname,
                'LNAME'     => (string)$lname,
                'ADDRESS'   => "",
                'PHONE'     => "",
            ),
        );
        
        $json = json_encode($array);
        
        // send a HTTP POST request with curl
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_USERPWD, 'user:' . $apiKey);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $httpCode;
    }
    
    /*private function remove_user_from_list( $listID, $email ) {
        $apiKey = wpjb_conf( "mailchimp_api_key" );
        
        // MailChimp API URL
        $memberID = md5( strtolower( $email ) );
        $dataCenter = substr($apiKey,strpos($apiKey,'-')+1);
        $url = 'https://' . $dataCenter . '.api.mailchimp.com/3.0/lists/' . $listID . '/members/' . $memberID;
        
        // member information
        $json = json_encode([
            'status'        => 'unsubscribed',
        ]);
        
        // send a HTTP POST request with curl
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_USERPWD, 'user:' . $apiKey);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return $httpCode;
    }*/
    

    function wpjb_mc_get_ids() {

        $request = Daq_Request::getInstance();
        $type = $request->post('type', false);
        $response = new stdClass();

        if($type == false) {
            $response->status = -5;
            $response->msg = __("Wrong type provided", 'wpjb-debugger');
            $response->ids = null;
            echo json_encode($response);
            wp_die();
        }

        $q = new Daq_Db_Query();
        $objects = $q->select(' COUNT(t.id) AS cnt')->from("Wpjb_Model_".ucfirst($type)." t")->fetchColumn();
        //$ids = array();

        /*foreach($objects as $obj) {
            $ids[] = $obj->id;
        }*/

        $response->status = 200;
        $response->msg = __("", 'wpjb-debugger');
        //$response->ids = json_encode($ids);
        $response->count = $objects;

        echo json_encode($response);
        wp_die();
    }

    
    function wpjb_mc_add_to_list() {

        $per_page = 25;

        $request = Daq_Request::getInstance();
        $type = $request->post('type', false);
        $records_number = $request->post('records_number', 0);
        $last = $request->post('last', 0);
        $response = new stdClass();
        $haveMore = ture;

        $class = "Wpjb_Model_" . ucfirst($type);

        if( ( $last + $per_page ) >= $records_number) {
            $haveMore = false;
        }

        $q = new Daq_Db_Query();
        $objects = $q->select()->from( $class . " t" )->limit( $per_page, $last )->execute();
        foreach($objects as $object) {

            if( $type == "resume" ) {
                $obj = new Wpjb_Model_Resume( $object->id );
                $listID = wpjb_conf( "mailchimp_candidate_list" );
                $email = $obj->getUser( true )->user_email;
                $fname = $obj->getUser( true )->first_name;
                $lname = $obj->getUser( true )->last_name;
            } else {
                $obj = new Wpjb_Model_Company( $object->id );
                $listID = wpjb_conf( "mailchimp_employer_list" );
                $user_info = get_userdata( $obj->user_id );
                $email = $user_info->user_email;
                $fname = $obj->company_name;
                $lname = "";
            }
            
            $this->add_user_to_list( $listID, $email, $fname, $lname );
            $last++;
        }

        $response->last = $last;
        $response->haveMore = $haveMore;
        $response->records_number = $records_number;

        echo json_encode($response);
        wp_die();
    }
    
    
    function load_admin_style() {
        
        wp_register_style( 'wpjobboard-mc-css', $this->get_baseurl() . "/assets/css/admin.css", false, time() );
        wp_enqueue_style( 'wpjobboard-mc-css');

        wp_register_script('wpjobboard-mc-refresh', $this->get_baseurl() . '/assets/js/refresh.js', array('jquery'), '1.0.0');
        wp_enqueue_script('wpjobboard-mc-refresh');
        wp_localize_script('wpjobboard-mc-refresh', 'wpjb_debugger_refresh_lang', array(
            "ajaxurl"   => admin_url('admin-ajax.php'),
        ) );
    }
    
    /**
     * Application Methods Updates Manager
     *
     * This functions is executed by plugins_loaded action
     *
     * @since 1.1
     * @return void
     */
    public function updates_manager() {

        if(!defined("Wpjb_Upgrade_Manager::MULTI")) {
            // old WPJB version without add-ons automatic updates support
            return;
        }

        $manager = new Wpjb_Upgrade_Manager(
            "wpjobboard-mailchimp/wpjobboard-mailchimp.php", 
            "wpjobboard-mailchimp", 
            "1.0.0"
        );
        $manager->connect();

        Wpjb_Project::getInstance()->env("upgrade")->{'wpjobboard-mailchimp'} = $manager;
    }
}
