<?php

class Wpjobboard_Ga_Frontend {

    /**
     * Frontend Constructor
     * 
     * Creates object which manages WPJB GA integration options.
     * 
     * @since 1.0
     * @return Wpjobboard_Ga_Frontend
     */
    public function __construct() {
        
        add_action( "init", array( $this, "init" ) );
    }
    
    /**
     * Frontend Init
     * 
     * Initiates frontend actions and filters
     * 
     * @since 1.0
     * @return void
     */
    public function init()
    {
        add_action( "wpjb_sh_manage_actions_right", array( $this, "manage_actions_right" ), 10, 2 );
        add_action( "wpjb_sh_manage_after", array( $this, "manage_after" ), 10, 2 );
        
        add_action( "wpjb_template_job_meta_text", array( $this, "object_stats" ) );
        
        wp_register_script(
            "wpjobboard-ga-chart",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/js/chart.bundle.min.js",
            array(),
            "1.0",
            true  
        );
        
        wp_register_script(
            "wpjobboard-ga",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/js/wpjobboard-ga.js",
            array( "jquery", "wpjobboard-ga-chart" ),
            "1.0",
            true  
        );
        
        wp_localize_script( "wpjobboard-ga", "wpjobboard_ga_lang", array(
            "ajaxurl" => admin_url( "admin-ajax.php" ),
            "compared_to" => __( "Compared to %d in the previous period.", "wpjobboard-ga" ),
            "no_changes" => __( "No changes compared to previous period", "wpjobboard-ga" ),
            "job_chart_title" => __( "Visits and Job Applications in the last 30 days.", "wpjobboard-ga" )
        ));
        
        wp_register_style( 
            "wpjobboard-ga",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/css/wpjobboard-ga.css",
            array( ),
            "1.0" 
        );
    }
    
    /**
     * Inserts an "Analytics" button 
     * 
     * The button is added into [wpjb_employer_panel] / Company Jobs
     * 
     * @param int $job_id       Job ID
     * @param int $post_id      Post ID
     */
    public function manage_actions_right($job_id, $post_id) {
        wp_enqueue_style( "wpjobboard-ga" );
        wp_enqueue_script( "wpjobboard-ga" );
        
        $html = new Daq_Helper_Html("a", array(
            "href" => "#",
            "class" => "wpjobboard-ga-analytics-button wpjb-manage-action",
            "data-post-id" => $post_id
        ),'<span class="wpjb-glyphs wpjb-icon-chart-bar"></span> ' . __("Analytics", "wpjobboard-ga"));
        
        echo $html->render();
    }
    
    /**
     * Displays chart in [wpjb_employer_panel] / Company Jobs
     * 
     * This function is called using wpjb_sh_manage_after filter
     * 
     * @param int $job_id       Job ID
     * @param int $post_id      Post ID
     */
    public function manage_after($job_id, $post_id = null) {
        include Wpjobboard_Ga::get_instance()->get_basedir() . "/includes/html-chart.php";
    }
    
    /**
     * Displays stats on job details page
     * 
     * This function is called using wpjb_template_job_meta_text filter
     * 
     * @see wpjb_template_job_meta_text filter
     * 
     * @param Daq_Db_OrmAbstract $object
     */
    public function object_stats($object) {
        include_once Wpjobboard_Ga::get_instance()->get_basedir() . "/includes/class-wpjobboard-ga-client.php";

        $cache_interval = apply_filters( "wpjobboard_ga_page_detail_cache", 3600 );
        $metric = apply_filters( "wpjobboard_ga_page_detail_metric", "pageviews" );

        $cache = get_post_meta( $object->post_id, "wpjobboard_ga_details", true );

        if( ! $cache ) {
            $ga = new Wpjobboard_Ga_Client();
            $ga->connect();
            $data = $ga->get_total($object->post_id)->getTotalsForAllResults();
            add_post_meta( $object->post_id, "wpjobboard_ga_details", array(
                "time" => time(),
                "data" => $data
            ), 1 );
        } elseif( isset( $cache["time"] ) && time()-$cache["time"] > $cache_interval ) {
            $ga = new Wpjobboard_Ga_Client();
            $ga->connect();
            $data = $ga->get_total($object->post_id)->getTotalsForAllResults();
            add_post_meta( $object->post_id, "wpjobboard_ga_details", array(
                "time" => time(),
                "data" => $data
            ), 1 );
        } else {
            $data = $cache["data"];
        }
        
        $metrics = array(
            "pageviews" => __( "Page Views", "wpjobboard-ga" ),
            "uniquePageviews" => __( "Unique Page Views", "wpjobboard-ga" ),
            "timeOnPage" => __( "Time on Page", "wpjobboard-ga" ),
            "bounces" => __( "Bounce Rate", "wpjobboard-ga" ),
            "entrances" => __( "Entrances", "wpjobboard-ga" )
        );
        
        $label = $metrics[$metric];
        $total = $data["ga:".$metric];
        
        include Wpjobboard_Ga::get_instance()->get_basedir() . "/includes/html-details.php";
    }
    
}