<?php

class Wpjobboard_Ga_Client 
{
    /**
     * Service configuration
     *
     * @var array
     */
    protected $_service = null;
    
    /**
     * Google Analytics Account ID
     *
     * @var string
     */
    protected $_account_id = null;
    
    /**
     * Google Analytics UA
     *
     * @var string
     */
    protected $_property_id = null;
    
    /**
     * Web Property Profile ID
     *
     * @var int
     */
    protected $_profile_id = null;
    
    /**
     * Google Analytics Object
     *
     * @var Google_Service_Analytics
     */
    protected $_analytics = null;
    
    /**
     * Constructor
     * 
     * Creates and instance of Wpjobboard_Ga_Client class.
     * 
     * @since 1.0
     * @return Wpjobboard_Ga_Client
     */
    public function __construct() {
        
        $basedir = Wpjobboard_Ga::get_instance()->get_basedir();
        include_once $basedir . '/google-api-php-client/vendor/autoload.php';
        
        $this->_service = get_option( "wpjobboard_ga_config" ) ;
        list($this->_account_id, $this->_property_id) = explode( "|", get_option( "wpjobboard_ga_uid" ) );
    }
    
    /**
     * Connects to user propery.
     * 
     * Establishes connection with selected website analytics.
     * 
     * @since 1.0
     * @return Google_Service_Analytics_Webproperty     or NULL if cannot connect
     */
    public function connect() {
        
        $service = get_option( "wpjobboard_ga_config" );
        
        $client = new Google_Client();
        $client->setApplicationName( $service["service_name"] );
        $client->setAuthConfig( json_decode( $service["service_key"], true ) );
        $client->setScopes(array(Google_Service_Analytics::ANALYTICS_READONLY));
        $this->_analytics = new Google_Service_Analytics($client);
        
        $profiles = $this->_analytics->management_profiles->listManagementProfiles($this->_account_id, $this->_property_id);

        if( isset( $profiles ) && $profiles[0] instanceof Google_Service_Analytics_Profile ) {
            $this->_profile_id = $profiles[0]->getId();
        }
        
        return $this->_profile_id;
    }
    
    /**
     * Returns Property
     * 
     * @since 1.0
     * @return Google_Service_Analytics_Webproperty
     */
    public function get_property()
    {
        return $this->_property;
    }
    
    /**
     * Get analytics data for the last 30 days
     * 
     * @param int $post_id                          ID of a post
     * @param boolean $live                         Return data including "today"
     * @return Google_Service_Analytics_GaData      Google analytics data
     */
    public function get_range_for_chart($post_id, $date_from, $date_to) 
    {
        $relative_url = trim( wp_make_link_relative( get_permalink( $post_id ) ), "/" );
        $page_path = apply_filters( "wpjobboard_ga_page_path", sprintf( "=/%s/", $relative_url ), $post_id );
        
        return $this->_analytics->data_ga->get(
            'ga:' . $this->_profile_id,
            $date_from,
            $date_to,
            'ga:visits',
            array(
                'filters' => 'ga:pagePath=' . $page_path,
                'dimensions' => 'ga:date',
                'metrics' => 'ga:pageviews,ga:uniquePageviews,ga:timeOnPage,ga:bounces,ga:entrances'
            )
        );
    }
    
    /**
     * Return total number of visits for a post identified with $post_id
     * 
     * @param int $post_id                          ID of a post for which data will be retrived
     * @param string $type                          Either "visits" or "pageviews"
     * @return Google_Service_Analytics_GaData      Google Analytics data
     */
    public function get_total($post_id, $type = "visits") 
    {
        list( $date, $time ) = explode( " ", get_post( $post_id )->post_date_gmt );
        return $this->get_range( $post_id, $type, $date, "today");
    }
    
    public function get_range($post_id, $type, $date_from, $date_to) 
    {
        $relative_url = trim( wp_make_link_relative( get_permalink( $post_id ) ), "/" );
        $page_path = apply_filters( "wpjobboard_ga_page_path", sprintf( "=/%s/", $relative_url ), $post_id );
        
        return $this->_analytics->data_ga->get(
            'ga:' . $this->_profile_id,
            $date_from,
            $date_to,
            'ga:' . $type,
            array(
                'filters' => 'ga:pagePath=' . $page_path,
                'dimensions' => 'ga:pagePath',
                'metrics' => 'ga:pageviews,ga:uniquePageviews,ga:timeOnPage,ga:bounces,ga:entrances'
            )
        );
    }
}
