<?php

class Wpjobboard_Ga_Ajax 
{
    protected $_object = null;
    
    public function __construct() {
        // Logged in users AJAX actions
        add_action( 'wp_ajax_wpjobboard_ga_data', array( $this, "data" ) );
    }
    
    /**
     * Get object ID from $_REQUEST["id"]
     * 
     * @return int  Object ID
     */
    protected function _get_object_id() {
        return absint( Daq_Request::getInstance()->getParam( "id" ) );
    }
    
    /**
     * Get object type from $_REQUEST["object_type"]
     * 
     * @return string   Object type, one of "job", "resume", "company".
     */
    protected function _get_object_type() {
        return Daq_Request::getInstance()->getParam( "object_type" );
    }
    
    /**
     * Get Object
     * 
     * @return Daq_Db_OrmAbstract
     */
    protected function _get_object() {
        $object_id = $this->_get_object_id();
        $object_type = $this->_get_object_type();
        
        if( $object_type == "job" ) {
            $object = new Wpjb_Model_Job( $object_id );
        } else if( $object_type == "resume" ) {
            $object = new Wpjb_Model_Resume( $object_id );
        } else if( $object_type == "company" ) {
            $object = new Wpjb_Model_Company( $object_id );
        } else {
            $object = null;
        }
        
        return $object;
    }
    
    /**
     * Return chart axis labels
     * 
     * @return array    Axis labels
     */
    protected function _get_chart_axis_labels() {
        $label = array(
            "x1" => __( "Unique Visits", "wpjobboard-ga" )
        );
        
        if( Daq_Request::getInstance()->getParam( "object_type" ) == "job" ) {
            $label["x2"] =  __( "Applications", "wpjobboard-ga" );
        }
        
        return $label;
    }
   
    /**
     * Returns formatted date
     * 
     * The date format is picked up based on date_format option.
     * 
     * @param string $date      Date in format YYYYMMDD
     * @return string           Date formatted according to settings in wp-admin / Settings panel.
     */
    protected function _format_date( $date ) {
        $year = substr( $date, 0, 4 );
        $month = substr( $date, 4, 2 );
        $day = substr( $date, 6, 2 );
        return date_i18n( get_option( "date_format"), strtotime( "$year-$month-$day" ) );
    }
    
    /**
     * Returns date range
     * 
     * For users without the edit_pages capability the range is limited to the last 30 days.
     * 
     * @param string $type      Either "from" or "to"
     * @return mixed            Array or a string date
     */
    protected function _get_date_range( $type = null ) {
        
        if( current_user_can( "edit_pages" ) ) {
            $date_from = Daq_Request::getInstance()->getParam( "date_from" );
            $date_to = Daq_Request::getInstance()->getParam( "date_to" );
        } else {
            $date_from = date( "Y-m-d", strtotime( "today -30 day" ) );
            $date_to = date( "Y-m-d", strtotime( "yesterday" ) );
        }
        
        if( $type == "from" ) {
            return $date_from;
        } else if( $type == "to" ) {
            return $date_to;
        } else {
            return array(
                "date_from" => $date_from,
                "date_to" => $date_to
            );
        }
    }
    
    /**
     * admin-ajax.php wpjobboard_ga_data action
     * 
     * This function is executed by wp_ajax_wpjobboard_ga_data actions when
     * user makes an AJAX request
     * 
     * @see wp_ajax_wpjobboard_ga_data filter
     * 
     * @return void
     */
    public function data() {
        include_once Wpjobboard_Ga::get_instance()->get_basedir() . "/includes/class-wpjobboard-ga-client.php";

        $object_id = $this->_get_object_id();
        $object_type = $this->_get_object_type();
        
        if( ! in_array( $object_type, array( "job", "resume", "company" ) ) ) {
            echo json_encode( array(
                "result" => 0,
                "message" => sprintf( __( "Incorrect object type [%s]", "wpjobboard-ga" ), $object_type )
            ));
            exit;
        }
        
        $object = $this->_get_object();
        $post = get_post( absint( $object->post_id ) );
        
        if( ! $post ) {
            echo json_encode( array(
                "result" => 0,
                "message" => sprintf( __( "Cannot find post [%s]", "wpjobboard-ga" ), $post->ID )
            ));
            exit;
        }
        
        $label = $this->_get_chart_axis_labels();
        $labels = array();
        $apps = array();
        $x1 = array();
        $x2 = array();
        $sum1 = 0;
        $sum2 = 0;

        $date_from = $this->_get_date_range( "from" );
        $date_to = $this->_get_date_range( "to" );
        
        $days = ( strtotime( $date_to ) - strtotime( $date_from ) ) / ( 3600 * 24 ) + 1;
        
        $previous_from = date( "Y-m-d", strtotime( "$date_from -$days DAY") );
        $previous_to = date( "Y-m-d", strtotime( "$date_to -$days DAY") );
        
        $rows = array();
        $rows_previous = array(); 
        
        $ga = new Wpjobboard_Ga_Client();
        $ga->connect();

        try {
            $data = $ga->get_range_for_chart($post->ID, $previous_from, $date_to);
        } catch(Google_Service_Exception $e) {
            $errs = $e->getErrors();
            if( ! $errs === null ) {
                $err = join( " / ", $errs[0] );
            } else {
                $err = print_r( json_decode( $e->getMessage() ), true ); 
            }

            echo json_encode( array(
                "result" => 0,
                "message" => sprintf( __( "Google Service Exception: %s", "wpjobboard-ga" ), $err )
            ));
            exit;
        } 
        
        if( $object_type == "job") {
            $apps = Wpjobboard_Ga::get_applications_for_chart($object->id);
        }
        
        $time_from = strtotime( $date_from );
        $time_to = strtotime( $date_to );
        
        $time_from_previous = strtotime( $previous_from );
        $time_to_previous = strtotime( $previous_to );
        
        foreach($data->rows as $row ) {
            $row_time = strtotime( $row[0] );
            if( $time_from <= $row_time && $time_to >= $row_time ) {
                $rows[] = $row;
            } else if( $time_from_previous <= $row_time && $time_to_previous >= $row_time ) {
                $rows_previous[] = $row;
            }
        }
        
        foreach($rows as $row) {
            $labels[] = $this->_format_date( $row[0] );
            $x1[] = (int)$row[2];
            $sum1 += (int)$row[2];
            if( isset( $apps[ $row[0] ] ) ) {
                $x2[] = $apps[$row[0]];
                $sum2 += $apps[$row[0]];
            } else {
                $x2[] = 0;
            }
        }
        
        $json = array(
            "result" => 1,
            "request_key" => "{$date_from}|{$date_to}",
            "label" => $label,
            "labels" => $labels,
            "x1" => $x1,
            "stats" => $this->_get_stats_from_rows( $rows, $rows_previous )
        );
        
        if( $object_type == "job" ) {
            $json["x2"] = $x2;
            $dates = array(
                "apps_current_from" => $date_from,
                "apps_current_to" => $date_to,
                "apps_previous_from" => $previous_from,
                "apps_previous_to" => $previous_to
            );
            $json["stats"]["applications"] = $this->_compile_applications($dates, $object_id);
        }

        echo json_encode( $json );
        exit;
    }
    
    /**
     * Returns stats for self::data().
     * 
     * Returns information about page views and unique sessions in a given time period
     * 
     * @param array $rows       Array with data for current date range
     * @param array $previous   Array with data for previous date range 
     * @return array            Page views and visits stats
     */
    protected function _get_stats_from_rows( $rows, $previous ) {

        $pv_current = 0;
        $pv_previous = 0;
        
        $un_current = 0;
        $un_previous = 0;
        
        $count = count( $rows );
        
        for( $i=0; $i<$count; $i++) {
            $pv_current += $rows[$i][1];
            $pv_previous += $previous[$i][1];
            
            $un_current += $rows[$i][2];
            $un_previous += $previous[$i][2];
        }
        
        $un_change = null;
        $pv_change = null;
        
        $stats = array();
        
        if( $un_previous > 0) {
            $un_change = round(100 * ($un_current-$un_previous) / $un_previous, 2 );
        }
        
        if( $pv_previous > 0 ) {
            $pv_change = round(100 * ($pv_current-$pv_previous) / $pv_previous, 2 );
        }
        
        $stats["unique"] = array(
            "current" => $un_current,
            "previous" => $un_previous,
            "change" => $un_change
        );
        
        $stats["pageviews"] = array(
            "current" => $pv_current,
            "previous" => $pv_previous,
            "change" => $pv_change
        );
        
        return $stats;
    }
    
    /**
     * Get applications stats for a given period
     * 
     * @param array $apps   Params for current and past applications search
     * @return array        Applications stats for given period
     */
    protected function _compile_applications( $apps, $job_id )
    {
        extract( $apps );

        $apps_current = Wpjb_Model_Application::search(array(
            "job" => $job_id,
            "date_from" => $apps_current_from,
            "date_to" => $apps_current_to,
            "count_only" => true
        ));

        $apps = array(
            "current" => $apps_current,
            "previous" => null,
            "change" => null
        );

        if( $apps_previous_from && $apps_previous_to ) {

            $apps_previous = Wpjb_Model_Application::search(array(
                "job" => $job_id,
                "date_from" => $apps_previous_from,
                "date_to" => $apps_previous_to,
                "count_only" => true
            ));

            $apps["previous"] = $apps_previous;
            
            if( $apps_previous ) {
                $apps_num = round(100 * ($apps_current-$apps_previous) / $apps_previous, 2 );
                $apps["change"] = $apps_num;
            } else if( $apps_previous == $apps_current ) {
                $apps["change"] = 0;
            } else if( $apps_current > 0 ) {
                $apps["change"] = 100;
            }
        }
        
        return $apps;
    }
}