<?php

class Wpjobboard_Ga_Admin {

    /**
     * Last returned Google Analytics error
     * 
     * This variable is being used only in configuration.
     *
     * @var string
     */
    public $last_ga_error = null;
    
    /**
     * Admin Constructor
     * 
     * Creates object which manages WPJB GA integration options.
     * 
     * @since 1.0
     * @return Wpjobboard_Ga_Admin
     */
    public function __construct() {
        
        add_action( "init", array( $this, "init" ) );
        add_action( "admin_menu", array( $this, "admin_menu" ) );
        add_action( "plugins_loaded", array( $this, "updates_manager" ) );
        
        add_action( "wpjb_page_sidebar", array( $this, "sidebar" ) );
    }
    
    /**
     * Application Methods Updates Manager
     *
     * This functions is executed by plugins_loaded action
     *
     * @since 1.1
     * @return void
     */
    public function updates_manager() {

        if(!defined("Wpjb_Upgrade_Manager::MULTI")) {
            // old WPJB version without add-ons automatic updates support
            return;
        }

        $manager = new Wpjb_Upgrade_Manager(
            "wpjobboard-google-analytics/wpjobboard-ga.php", 
            "wpjobboard-google-analytics", 
            "1.0.0"
        );
        $manager->connect();

        Wpjb_Project::getInstance()->env("upgrade")->{'wpjobboard-google-analytics'} = $manager;
    }
    
    /**
     * Admin Init
     * 
     * Initiates wp-admin actions and filters
     * 
     * @since 1.0
     * @return void
     */
    public function init() {
       
        wp_register_script(
            "wpjobboard-ga-chart",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/js/chart.bundle.min.js",
            array(),
            "1.0",
            true  
        );
        
        wp_register_script(
            "wpjobboard-ga-admin",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/js/wpjobboard-ga-admin.js",
            array( "jquery", "jquery-ui-dialog", "jquery-ui-datepicker", "wpjobboard-ga-chart" ),
            "1.0",
            true  
        );
        
        wp_localize_script( "wpjobboard-ga-admin", "wpjobboard_ga_admin_lang", array(
           "compared_to" => __( "Compared to %d in the previous period.", "wpjobboard-ga" ),
           "no_changes" => __( "No changes compared to previous period", "wpjobboard-ga" )
        ));
        
        wp_register_style(
            'wpjobboard-ui-datepicker', 
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/css/jquery-ui-date-picker.css",
            array(),
            "1.0"
        );
        
        wp_register_style(
            "wpjobboard-ga-admin",
            Wpjobboard_Ga::get_instance()->get_baseurl() . "/assets/css/wpjobboard-ga-admin-sidebar.css",
            array( "wpjobboard-ui-datepicker", "wp-jquery-ui-dialog" ),
            "1.0"
        );
    }
    
    /**
     * Adds menu option in wp-admin / Options panel
     * 
     * This function is being called by admin_menu action
     * 
     * @see admin_menu action
     * 
     * @since 1.0
     * @return void
     */
    public function admin_menu() {
        add_options_page( 
            __( "WPJobBoard - Google Analytics Integration", "wpjobboard-ga"), 
            __( "WPJobBoard - GA", "wpjobboard-ga" ),
            "administrator", 
            "wpjobboard-ga", 
            array( $this, "options_page" )
        );
    }
    
    /**
     * Displays configuration page
     * 
     * @since 1.0
     * @return void
     */
    public function options_page() {
        
        $basedir = Wpjobboard_Ga::get_instance()->get_basedir();
        $form_error = null;
        
        include_once $basedir . '/google-api-php-client/vendor/autoload.php';
        include_once $basedir . '/includes/class-wpjobboard-ga-form-config.php';
        
        $form = new Wpjobboard_Ga_Form_Config();
        
        if( isset( $_GET["reset-all"] ) && $_GET["reset-all"] == "1" ) {
            delete_option( "wpjobboard_ga_config" );
            delete_option( "wpjobboard_ga_uid" );
            wp_redirect( remove_query_arg( array( "noheader", "reset-all" ) ) );
        }
        
        if( isset( $_POST["wpjobboard_ga_config"] ) ) {
            if( ! $form->isValid( stripslashes_deep( $_POST ) ) ) {
                $form_error = __( "There are errors in your form.", "wpjobboard-ga" );
            } else if( ! $this->validate_ga_access( stripslashes_deep( $_POST ) ) ) {
                $form_error = $this->last_ga_error;
            } else {
                update_option( "wpjobboard_ga_config", $form->getValues() );
            }
        }
        
        $service = get_option( "wpjobboard_ga_config" );
        $service_uid = get_option( "wpjobboard_ga_uid" );
        
        if( $service ) {
            $options = $this->get_ga_options($service);
        }
        
        if( isset( $_POST["wpjobboard_ga_uid"] ) && isset( $_POST["service_uid"] ) ) {
            update_option( "wpjobboard_ga_uid", $_POST["service_uid"]);
            $service_uid = $_POST["service_uid"];
        }
        
        include $basedir . '/admin/config.php';
    }
    
    /**
     * Check if user has valid GA Access
     * 
     * @since 1.0
     * @param array $service    Analytics service access
     * @return boolean          True if can connect with GA
     */
    public function validate_ga_access( $service ) 
    {
        try {
            $this->get_ga_profiles( $service );
        } catch(Google_Service_Exception $e) {
            
            $errs = $e->getErrors();
            if( ! $errs === null ) {
                $err = join( " / ", $errs[0] );
            } else {
                $err = print_r( json_decode( $e->getMessage() ), true ); 
            }
            $this->last_ga_error = sprintf( __( "Google Service Exception: %s", "wpjobboard-ga" ), $err );
            return false;
        } catch(Exception $e) {
            $this->last_ga_error = $e->getMessage();
            return false;
        }
        
        return true;
    }
    
    /**
     * Returns Google Analytics Profiles
     * 
     * @param array $service
     * @return object
     */
    public function get_ga_profiles( $service ) 
    {
        
        $client = new Google_Client();
        $client->setApplicationName( $service["service_name"] );
        $client->setAuthConfig( json_decode( $service["service_key"], true ) );
        $client->setScopes(array(Google_Service_Analytics::ANALYTICS_READONLY));
        $analytics = new Google_Service_Analytics($client);

        $accounts = $analytics->management_accounts->listManagementAccounts();
        $properties = $analytics->management_webproperties->listManagementWebproperties($accounts[0]->id);

        return $properties->getItems();
    }
    
    /**
     * Returns GA accounts and properties
     * 
     * @param array $service
     * @return array
     */
    public function get_ga_options( $service ) {
        
        $client = new Google_Client();
        $client->setApplicationName( $service["service_name"] );
        $client->setAuthConfig( json_decode( $service["service_key"], true ) );
        $client->setScopes(array(Google_Service_Analytics::ANALYTICS_READONLY));
        $analytics = new Google_Service_Analytics($client);

        $options = array();
        $accounts = $analytics->management_accounts->listManagementAccounts();
        
        foreach($accounts as $account) {
            $options[$account->id] = array(
                "id" => $account->id,
                "name" => $account->name,
                "properties" => array()
            );
            
            $properties = $analytics->management_webproperties->listManagementWebproperties($account->id);
            
            foreach($properties as $property) {
                $options[$account->id]["properties"][] = array(
                    "id" => $property->id,
                    "name" => $property->name,
                    "website_url" => $property->websiteUrl
                );
            }
            
        }

        return $options;
    }
    
    /**
     * Displays widget with stats in wp-admin / Job Board sidebar
     * 
     * @param Daq_Form_Abstract $form
     */
    public function sidebar($form) {
        
        $id = $form->getObject()->id;
        
        if( $form->getObject() instanceof Wpjb_Model_Job ) {
            $object_type = "job";
        } else if( $form->getObject() instanceof Wpjb_Model_Resume ) {
            $object_type = "resume";
        } else if( $form->getObject() instanceof Wpjb_Model_Company ) {
            $object_type = "company";
        } else {
            return;
        }
        
        include_once Wpjobboard_Ga::get_instance()->get_basedir() . "/includes/class-wpjobboard-ga-client.php";

        $ga = new Wpjobboard_Ga_Client();
        $ga->connect();

        $post = get_post( $form->getObject()->post_id );
        
        wp_enqueue_script( "wpjobboard-ga-admin" );
        
        wp_enqueue_style( "wp-jquery-ui-dialog");
        wp_enqueue_style( "wpjobboard-ga-admin" );
        
        $range = apply_filters( "wpjobboard_ga_config_range", "last30" );
        

        
        include Wpjobboard_Ga::get_instance()->get_basedir() . '/includes/html-admin-widget.php';
    }
}
