<?php

// Bail if called directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Wpjobboard_Ca
{
    /**
     * Wpjobboard_Ca Singleton
     * 
     * @var Wpjobboard_Ca
     */
    protected static $_instance = null;
    
    /**
     * Path to wpjobboard-ca (eg. /public_html/wp-content/plugins/wpjobboard-ca)
     *
     * @var string
     */
    protected $_basedir = null;
    
    /**
     * URL to wpjobboard-ca (eg. http://example.com/wp-content/plugins/wpjobboard-ca)
     *
     * @var string
     */
    protected $_baseurl = null;
    
    /**
     * Refresh Object
     * 
     * @var Wpjobboard_Ca_Refresh
     */
    protected $_refresh = null;

    
    /**
     * Object contructor
     * 
     * Protected constructor allows to create only one instance of this class.
     * 
     * @since 1.0
     * @return Wpjobboard_Ca
     */
    protected function __construct() {
        
        $this->_basedir = dirname( dirname( __FILE__ ) );
        $this->_baseurl = plugin_dir_url( dirname( __FILE__ ) );
        
        include_once $this->_basedir . "/includes/class-wpjobboard-ca-refresh.php";
        $this->_refresh = new Wpjobboard_Ca_Refresh();
        
        add_action( "plugins_loaded", array( $this, "updates_manager" ) );
        add_action( "init", array( $this, "init" ));
        
        add_action( 'admin_enqueue_scripts',  array( $this, 'load_admin_style' ) );
    }
    
    /**
     * Singleton
     * 
     * Creates if NULL and returns Wpjobboard_Ca instance
     * 
     * @since 1.0
     * @return Wpjobboard_Ca
     */
    public static function get_instance() {
        if( self::$_instance === null ) {
            self::$_instance = new self();
        }
        
        return self::$_instance;
    }
    
    /**
     * Returns basedir path
     * 
     * @since 1.0
     * @return string   Basedir path (eg. /public_html/wp-content/plugins/wpjobboard-ca)
     */
    public function get_basedir() {
        return $this->_basedir;
    }
    
    /**
     * Returns plugin base URL
     * 
     * @since 1.0
     * @return string   Basedir path (eg. http://example.com/wp-content/plugins/wpjobboard-ca)
     */
    public function get_baseurl() {
        return $this->_baseurl;
    }
    
    /**
     * General Init Function
     * 
     * This function is executed in both frontend and wp-admin
     * 
     * @since 1.0
     * @return void
     */
    public function init() {
        
        add_filter( "wpjb_config_sections", array( &$this, "wpjb_config_sections" ) );
        
        add_filter( 'the_title', array( &$this, "anonymize_title" ) );
        add_filter( 'wp_title', array( &$this, "anonymize_title" ) );
        add_filter( 'wpjb_candidate_name', array( &$this, "anonymize_name" ), 10, 2);
        
        add_filter( 'wpjb_generate_slug', array( &$this, "generate_anonymous_slug" ), 10, 4 );
        
        add_action( 'wp_head', array( &$this, "hide_candidate_search_engines" ) );
    }
    
    /**
     * ACA Updates Manager
     *
     * This functions is executed by plugins_loaded action
     *
     * @since 1.1
     * @return void
     */
    public function updates_manager() {

        if(!is_admin()) {
            return;
        }

        if(!defined("Wpjb_Upgrade_Manager::MULTI")) {
            // old WPJB version without add-ons automatic updates support
            return;
        }

        $manager = new Wpjb_Upgrade_Manager(
            "wpjobboard-candidate-anonymizer/wpjobboard-ca.php", 
            "wpjobboard-candidate-anonymizer", 
            "1.0.0"
        );
        $manager->connect();

        Wpjb_Project::getInstance()->env("upgrade")->{'wpjobboard-candidate-anonymizer'} = $manager;
    }

    /**
     * Adds config option in Seetings (WPJB)
     * 
     * @param Array $config
     * @return Array
     */
    function wpjb_config_sections($config) {
        
        include_once $this->get_basedir() . "/includes/class-wpjobboard-ca-form.php";
        
        $config[1]['item'][] = array(
            "form"  => "anonymizer",
            "class" => "Wpjobboard_Ca_Config_Form",
            "icon"  => "wpjb-icon-user-secret",
            "title" => __("Candidate Anonymizer", "wpjobboard-ca"),
            "order" => 10,
        );

        return $config;
    }
    
    /**
     * Loads CSS & JavaScript 
     */
    function load_admin_style() {
        
        wp_register_style( 'wpjobboard-ca-css', $this->get_baseurl() . "/assets/css/admin.css", false, time() );
        wp_enqueue_style( 'wpjobboard-ca-css');

        wp_register_script('wpjobboard-ca-refresh', $this->get_baseurl() . '/assets/js/refresh-ca.js', array('jquery'), '1.0.0');
        wp_enqueue_script('wpjobboard-ca-refresh');
        wp_localize_script('wpjobboard-ca-refresh', 'wpjb_debugger_refresh_lang', array(
            "ajaxurl"   => admin_url('admin-ajax.php'),
        ) );
        
        wp_register_script('wpjobboard-ca-js-main', $this->get_baseurl() . '/assets/js/main-ca.js', array('jquery'), '1.0.0');
        wp_enqueue_script('wpjobboard-ca-js-main');
        wp_localize_script('wpjobboard-ca-js-main', 'wpjb_debugger_refresh_lang', array(
            "ajaxurl"   => admin_url('admin-ajax.php'),
        ) );
    }
    
    function anonymize_title($title) {
        
        if( is_admin() ) {
            return $title;
        }

        global $post;

        if($post->post_type != "resume") {
            return $title;
        }  

        $q = new Daq_Db_Query();
        $resume = $q->select()->from("Wpjb_Model_Resume t")
                                 ->where("t.post_id = ?", $post->ID)
                                 ->execute();

        $resume_id = $resume[0]->id;

        $can_see = $this->can_see_full_name( $resume_id );
        if( $can_see ) {
            return $title;
        }     

        if( $resume[0]->getSearch(true)->fullname != $title && strpos( $title, $resume[0]->getSearch(true)->fullname ) !== 0 ) {
            return $title;
        }
        
        $split_name = explode(" ", trim($title));
        $return_name = trim($split_name[0]) . " ";

        for($i = 1 ; $i < count($split_name) ; $i++) {
            if( $i == 1) {
                $return_name .= substr(trim($split_name[$i]), 0, 1) . " ";
            } else {
                $return_name .= " " . $split_name[$i];
            }
        }

        return trim($return_name);    
    }
    
    function anonymize_name($name, $resume_id) {

        $can_see = $this->can_see_full_name( $resume_id );
        if( $can_see ) {
            return $name;
        }

        $split_name = explode(" ", trim($name));
        $return_name = trim($split_name[0]) . " ";

        for($i = 1 ; $i < count($split_name) ; $i++) {
            $return_name .= substr(trim($split_name[$i]), 0, 1) . ". ";
        }

        return trim($return_name);
    }
    
    function generate_anonymous_slug($slug, $model, $title, $id) {

        if($model == "resume") {
            $slug = $this->generate_slug_pattern( $id );
        }

        return $slug;
    }
    
    function hide_candidate_search_engines() {
        
        if ( ! is_singular( 'resume' ) ) {
            return;
        }
        
        $restrictions = wpjb_conf( "wpjobboard_ca_robots", null );
        if( $restrictions == null ) {
            return;
        }

        //$title = the_title_attribute( [ 'echo' => FALSE ] );
        echo '<meta name="robots" content="'.join( ", ", $restrictions).'" />' . PHP_EOL;
    }

    private function can_see_full_name( $resume_id ) {
        
        $hide = wpjb_conf( "wpjobboard_ca_hide", array( 0 ) );

        if( $hide[0] != 1 ) {
            return true;
        }   
        
        // Admin can see full name
        if( current_user_can('manage_options') ) {
            return true;
        }

        $cv_access = wpjb_conf( "wpjobboard_hide_exception", array() );
        // Signed users can see full name
        if( in_array( 1, $cv_access ) && is_user_logged_in() ) {
            return true;
        }

        // candidate can always see his name
        if(Wpjb_Model_Resume::current() && Wpjb_Model_Resume::current()->id == $resume_id) {
           return true;
        }
        
        $premium = false;
        if(wpjr_has_premium_access($resume_id)) {
            // if has valid hash, always allow
            $premium = true;
        }
        if(Wpjb_Model_Company::current() && Wpjb_Model_Company::current()->canViewResume($resume_id)) {
            // employer received at least one application from this candidate
            // and employers can view full applicants resumes
            // this option is enabled in wp-admin / Settings (WPJB) / Resumes Options panel
            $premium = true;
        }
        if(current_user_can('manage_options')) {
            // admin can see anything
            $premium = true;
        }
        
        // Premium users can see full name
        if( in_array( 2, $cv_access ) && $premium ) {
            return true;
        }
        
        return false;
    }
    
    public function generate_slug_pattern( $resume_id ) {
        
        $pattern =  wpjb_conf( "wpjobboard_ca_slug", 0 );
            
        //$form = new Wpjb_Form_Resume( $id );
        $form = new Wpjb_Form_Admin_Resume( $resume_id );
        foreach( $form->getFields() as $field ) {
            $pattern = str_replace( '{$' . $field->getName() . '}', $field->getValue(), $pattern);
        }
        
        $pattern = str_replace( '{$id}', $resume_id, $pattern);
        $pattern = str_replace( '{$first_letter_of_last_name}', substr( $form->getElement("last_name")->getValue(), 0, 1), $pattern);
        $pattern = str_replace( '{$random}', wp_generate_password(), $pattern);
        $pattern = sanitize_title($pattern);
        $pattern = preg_replace("([^A-z0-9\-]+)", "", $pattern);
        $slug = strtolower( $pattern);
        
        return apply_filters( "wpjb_ca_slug", $slug, $resume_id );
    }

}
